/*****************************************************************************
*  LPC11U6x RC5 transmit demo program
*
*  1. Use SCT1 timer to send encoded RC5 frames @ SCT1_OUT0 (P2_16)
*  2. Use S2 (P0_16) key to send key code
*  3. Use MRT to generate timeout delay between RC5 frames
*****************************************************************************/

#include "board.h"
#include "global.h"

#define RC5_DATA0()      Chip_IOCON_PinMuxSet(LPC_IOCON, 2, 8, IOCON_MODE_PULLDOWN);	// input low (pull down)
#define RC5_DATA1()      Chip_IOCON_PinMuxSet(LPC_IOCON, 2, 8, IOCON_MODE_PULLUP); 	// input high (pull up)

static uint8_t  state;                                 // driver state
static uint8_t  count;                                 // bit counter
static uint8_t  flag;                                  // frame send ready flag
static uint16_t frame;                                 // frame

void RC5_Send(uint16_t fr)
{
    frame = fr << 2;                                   // MSB first, 14 bits shift left 2
    state = 0;
    flag  = 0;                                         // clear ready flag
    count = 14;                                        // set bit count to 14

    Chip_SCT_EnableEventInt(LPC_SCT1 , SCT_EVT_4 );    // event 4 generates interrupt

    while (!flag) ;                                    // wait for ready flag
}

void SCT0_1_IRQHandler(void)
{
    if (count)                                         // send next bit if required
    {
        if (state)
        {
            if (frame & 0x8000)
            {
                RC5_DATA1();
            }
            else
            {
                RC5_DATA0();
            }
            state = 0;
            count --;
            frame <<= 1;
        }
        else
        {
            if (frame & 0x8000)
            {
                RC5_DATA0();
            }
            else
            {
                RC5_DATA1();
            }
            state = 1;
        }
    }
    else
    {
        RC5_DATA0();                                       	// inactive carrier pulse

        Chip_SCT_DisableEventInt(LPC_SCT1,  (CHIP_SCT_EVENT_T) 0x00);			// disable interrupt
        flag = 1;                                          	// signal ready
    }
    LPC_SCT1->EVFLAG =	SCT_EVT_4;							// clear event 4 flag
}

void RC5_Init(void)
{
    RC5_DATA0();                                           					// SCT0_IN0 low

    Chip_SCT_Init(LPC_SCT1);												// enable the SCT0 clock

/********************************************************************
*  SCT_L: low part configuration:
********************************************************************/

    Chip_SCT_SetMatchCountL(LPC_SCT1, SCT_MATCH_0, (SystemCoreClock / 48000)-1);		// 75% of 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT1, SCT_MATCH_0, (SystemCoreClock / 48000)-1);

    Chip_SCT_SetMatchCountL(LPC_SCT1, SCT_MATCH_1, (SystemCoreClock / 36000)-1);		// 36 KHz
    Chip_SCT_SetMatchReloadL(LPC_SCT1, SCT_MATCH_1, (SystemCoreClock / 36000)-1);


    Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_0, 0x00000003);				// event 0 happens in all states
    Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match 0
                                                                          SCT_HEVENT_L		    |	// HEVENT   [4]     = use L state & match
                                                                          SCT_OUTSEL_L		    |	// OUTSEL   [5]     = select input from IOSEL
                                                                          SCT_IOSEL_RISEIN0	  |	// IOSEL    [9:6]   = select input 0
                                                                          SCT_IOCOND_HIGH	    |	// IOCOND   [11:10] = high level
                                                                          SCT_COMBMODE_AND	  |	// COMBMODE [13:12] = match AND IO condition
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    |	// STATEV   [19:15] = no state change
                                                                          SCT_MATCHMEM		    |	// MATCHMEM [20]    = "equal" to match
                                                                          SCT_DIRECTION		    ));	// DIRECTION[22:21] = direction independent


    Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_1, 0x00000003);				// event 1 happens in all states
    Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH1	|	// MATCHSEL [3:0]   = related to match 1
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_1);			// IR LED low  @ event 0
    Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_0);				// IR LED high @ event 1

    LPC_SCT1->LIMIT_L        = SCT_EVT_1;                   			// events 1 is used as counter limit

/********************************************************************
*  SCT_H: high part configuration:
********************************************************************/

    Chip_SCT_SetMatchCountH(LPC_SCT1, SCT_MATCH_0, (SystemCoreClock / 1125)-1);		// 36 KHz / 32 ticks = 1125
    Chip_SCT_SetMatchReloadH(LPC_SCT1, SCT_MATCH_0, (SystemCoreClock / 1125)-1);


    Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_4, 0x00000003);				// event 4 only happens in all states
    Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL [3:0]   = related to match_H 0
                                                                          SCT_HEVENT_H		    |	// HEVENT   [4]     = use H state & match
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE [13:12] = match condition only
                                                                          SCT_STATELD_0		    |	// STATELD  [14]    = STATEV is added to state
                                                                          SCT_STATEEV_0		    ));	// STATEV   [19:15] = no state change


    LPC_SCT1->LIMIT_H        = SCT_EVT_4;                   					// event 4 is used as H counter limit

    NVIC_EnableIRQ(SCT0_1_IRQn);                             					// enable SCT interrupt

    Chip_SCT_ClearControl(LPC_SCT1,(SCT_CTRL_HALT_L |SCT_CTRL_HALT_H));      // unhalt both low and high counter

}

int main(void)
{
  static uint8_t  trg = 1;
  static uint8_t  tgl = 1;
  static uint16_t frame;

  SystemCoreClockUpdate();
  Board_Init();


  Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_IOCON);		    		//enable IOCON clock

  Chip_IOCON_PinMuxSet(LPC_IOCON, 2,8  , 0x00000081);				// P2_8  is SCT1_IN0
  Chip_IOCON_PinMuxSet(LPC_IOCON, 2,16 , 0x00000081);				// P2_16 is SCT1_OUT0

  RC5_Init();
  SysTick_Config(SystemCoreClock/8);                     			// SysTick interrupt @ 1/8 sec

    while (1)
    {
        frame = (1   << 13) |                              			// set start (S) bit
                (1   << 12) |                              			// set field (F) bit
                (tgl << 11) |                             	 		// clr control / toggle (C) bit
                (20  << 6 ) |                              			// system  = 20
                (9   << 0 );                               			// command = 9

        if (!Chip_GPIO_GetPinState(LPC_GPIO, 0, 16))                // P0_16 high to low?
        {
            if (trg)
            {
                trg = 0;                                   			// trigger P0_16 low
            }
            RC5_Send(frame);
        }
        else if ((Chip_GPIO_GetPinState(LPC_GPIO,0, 1)) && !trg)    // P0_1 low to high?
        {
            trg = 1;                                       			// trigger P0_1 high
            tgl = !tgl;                                    			// toggle RC5 control bit
        }
        __WFI();
    }
}
